// $Id: twoq.cc,v 1.6 1998/11/29 18:38:46 islene Exp $

/* Copyright 1998 Alexandre Oliva <oliva@dcc.unicamp.br>, Islene Calciolari Garcia <islene@dcc.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Pallottino (TWO-Q): This algorithm [GoldberbRadzik93] uses two FIFO
 * queues, one of low priority and another of high priority. Each
 * labeled node appears on exactly one of the queues. The next node to
 * be scanned is removed from the head of the high priority queue if
 * the queue is no empty and from the head of the low priority queue
 * otherwise. A node that becomes labeled is added to the tail of the
 * low priority queue if this is the first time this node became
 * labeled, and to the tail of the high priority queue otherwise.
 */

#include "spalgo.h"
#include "graph.h"
#include "node.h"
#include "basic_node.h"

#include <queue>

class pallottino :
  public shortest_path_algorithm<pallottino, graph< basic_node > > {
  /** Convenience type aliases.  */
  typedef shortest_path_algorithm<pallottino, graph< basic_node > > inherited;
  typedef std::queue<node_t*> queue_t;

  /** Queues of nodes to scan; high and low priorities.  */
  queue_t to_scan_high, to_scan_low;

  /** Inserts the node in the high- or in the low-priority queue,
      depending on whether it had already been reached or not.  */
  void push(node_t& node, bool was_reached = true) {
    if (was_reached)
      to_scan_high.push(&node);
    else
      to_scan_low.push(&node);
  }

  /** Returns true if there are no more nodes left to scan in the
      high-priority queue.  */
  bool high_empty() const {
    return to_scan_high.empty();
  }

  /** Returns true if there are no more nodes left to scan in the
      low-priority queue.  */
  bool low_empty() const {
    return to_scan_low.empty();
  }

  /** Returns true if there are no more nodes left to scan.  */
  bool empty() const {
    return high_empty() && low_empty();
  }

  /** Removes a node from the front of one of the queues.  */
  node_t *pop() {
    queue_t *queue;
    if (!high_empty())
      queue = &to_scan_high; // use the high-priority queue
    else if (!low_empty())
      queue = &to_scan_low; // use the low-priority queue
    else
      abort(); // cannot happen
    queue_t &to_scan = *queue; // use the reference notation, like the
                               // other algorithms.
    node_t *node = to_scan.front();
    to_scan.pop();
    return node;
  }

public:
  /** Initialize the queue with the source node.  */
  pallottino(graph_t& g) : inherited(g) {
    push(g.source()); // scan source first
  }

  /** If queue is empty, return NULL, otherwise, pop() the first
      element.  */
  node_t *select() {
    if (empty())
      return 0;
    return pop();
  }

  /** Return true iff the node was relabeled.  If it has become
      labeled, it is inserted in the queue.  */
  bool test_relabel(arc_t& arc) {
    bool was_labeled = arc.to().is_labeled();
    bool was_reached = arc.to().is_reached();
    bool relabeled = inherited::test_relabel(arc);
    // was_labeled implies it was in the queue already
    if (relabeled && !was_labeled)
      push(arc.to(), was_reached); // scan it sooner or later
    return relabeled;
  }
};

typedef pallottino algorithm_t;

#include "main.h"
