#ifndef _graph_h
#define _graph_h

// $Id: graph.h,v 1.12 1998/11/29 05:34:53 oliva Exp $

/* Copyright 1998 Alexandre Oliva <oliva@dcc.unicamp.br>, Islene Calciolari Garcia <islene@dcc.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <vector>
#include <string>
#include <iostream>

/** This is the same as getline, but string-ized getline is not
    implemented in the current libstdc++ :-( */
inline std::string read_to_eol(std::istream& in, char eol = '\n') {
  std::string s;
  char ch;
  while(in.get(ch) && ch != eol)
    s += ch;
  return s;
}

/** Discard the rest of the current line.  */
inline std::istream& skip_to_eol(std::istream& in, char eol = '\n') {
  read_to_eol(in, eol);
  return in;
}

/** A graph is, essentially, a container of nodes.  Any (unlikely)
    alternate implementation should define nested types graph_t,
    node_t, distance_t, node_id_t, adjlist_t and arc_t.  Two
    constructors must be available, one that constructs an empty graph
    and another that reads a graph from a stream.  Operators to read a
    graph from a stream and write the solution to a stream must be
    provided.  Operations such as get_title(), arc_count(), source()
    and sum_dist() must be implemented.  */
template <class node_T>
class graph : public std::vector<node_T> {
 public:
  /** The graph type.  */
  typedef graph graph_t;

  /** The node type.  */
  typedef node_T node_t;

  /** The distance type.  */
  typedef typename node_t::distance_t distance_t;

  /** The type of the adjacency list.  */
  typedef typename node_t::adjlist_t adjlist_t;

  /** The arc type.  */
  typedef typename node_t::arc_t arc_t;

  /** The node_id type.  */
  typedef typename node_t::node_id_t node_id_t;

  /** Constructs an empty graph.  */
  graph() : s(0) {}
  /** Reads a graph from the standard input, in the extended DIMACS
      format or in the non-standard binary-compressed format.  */
  graph(std::istream& in) : title(), s(0), arcs(0) { in >> *this; }

 private:
  /** The ``name'' of the graph.  */
  std::string title;
 public:
  /** Obtains the name of the graph.  */
  std::string const& get_title() const { return title; }
  /** Modifies the name of the graph.  */
  void set_title(const std::string& new_title) { title = new_title; }

 private:
  /** Counts the total number of arcs of the graph.  */
  unsigned arcs;
 public:
  /** Returns the total arc count.  */
  unsigned arc_count() { return arcs; }

 private:
  /** Points to the source node.  */
  node_t *s;
 public:
  /** Returns a reference to source node.  */
  node_t& source() const { return *s; }

  /** Reserve space for `a' arcs on the node whose node-id is `n'.
      This will just speed up the construction of the adjacency list
      of the node, it is not required.  */
  void prealloc_arcs(node_id_t n, unsigned a) {
    (*this)[n-1].adjlist().reserve(a);
  }

  /** Create `n' nodes and pre-allocate space for `ceiling(a/n)' arcs
      for each one.  */
  void problem(unsigned n, unsigned a) {
    a += n; a /= n; // calculate the average per-node arc count.
    clear(); // make sure it is empty.
    arcs = 0;
    reserve(n); // pre-allocate space for the nodes, as an optimization.
    for(unsigned i = 0; i < n; ++i) {
      push_back(node_t(i+1)); // insert node i+1
      prealloc_arcs(i+1, a);
    }
  }

  /** Mark the node whose node_id is `n' as the source node.  */
  void source(node_id_t n) {
    if (n > size()) {
      cerr << "source node (" << n << ") out of range" << endl;
      abort();
    }
    s = &(*this)[n-1]; // n-1, not n, because we start at 0
  }

  /** Add an arc with tail on from_id, head on to_id and the given length */
  void arc(node_id_t from_id, node_id_t to_id, distance_t length) {
    if (to_id > size() || from_id > size()) {
      cerr << "node out of range" << endl;
      abort();
    }

    ++arcs;

    node_t& from = (*this)[from_id-1]; // find tail node
    node_t& to = (*this)[to_id-1]; // find head node
    from.adjlist().push_back(arc_t(from, to, length)); // append arc
  }
  
 public:
  /** This type is the header of a binary-compressed file.  */
  struct header_t { unsigned n, a, source; };
  /** This type contains the arc information in a binary-compressed file.  */
  struct arc_data_t { unsigned to; distance_t length; };

  /* Read a graph from the given stream, in extended DIMACS format or
     in binary-compressed format.  */
  friend std::istream& operator>>(std::istream& in, graph_t& g) {
    for(char c; in >> c;) {
      switch (c) {
      case 'b': { // extension: read in binary format
	header_t header;
	in.read(&header, sizeof(header));
	// we don't want to pre-allocate arcs now, so pass arcs as 0
#if ! JUST_READ
	g.problem(header.n, 0);
	g.source(header.source);
#endif
	for(node_id_t from = 1; from <= header.n; ++from) {
	  unsigned arcs;
	  in.read(&arcs, sizeof(arcs));
#if ! JUST_READ
	  g.prealloc_arcs(from, arcs);
#endif
	  for(unsigned base = 0, size = 0; base < arcs; base += size) {
	    // read 8 Kb chunks to try to speed things up
	    size = min(arcs-base, unsigned(8192 / sizeof(arc_data_t)));
	    arc_data_t arc_data[size];
	    in.read(&arc_data, sizeof(arc_data));
#if ! JUST_READ
	    for (unsigned i = base; i < base+size; ++i)
	      g.arc(from, arc_data[i].to, arc_data[i].length);
#endif
	  }
	}
	break;
      }
	
      case 'c': // comment: c ...
	skip_to_eol(in);
	break;

      case 't': // title:   t ...
	std::ws(in); // skip blanks
	g.set_title(read_to_eol(in));
	break;

      case 'p': { // problem: p TYPE NODES ARCS
	char ch; in >> ch; in >> ch; // ignore problem type
	unsigned n; in >> n; // read NODES
	unsigned a; in >> a; // read ARCS
#if ! JUST_READ
	g.problem(n, a);
#endif
	// Note that we're translating the range [1,n] to [0,n-1]
	skip_to_eol(in);
	break;
      }
      
      case 'n': { // source:  n SOURCE
	unsigned n; in >> n; // read SOURCE
#if ! JUST_READ
	g.source(n);
#endif
	skip_to_eol(in);
	break;
      }

      case 'a': { // arc:     a TAIL HEAD LENGTH
	unsigned from, to; distance_t length;
	in >> from >> to >> length;

#if ! JUST_READ
	g.arc(from, to, length);
#endif

	skip_to_eol(in); // skip to end of line
	break;
      }

      default:
	cerr << "parse error" << endl;
	abort();
      }
    }
    if (!g.s) {
      cerr << "source node not specified" << endl;
      abort();
    }
    return in;
  }

  /** Output the solution of the shortest path problem to the given
      stream.  For each node, its distance from the source and its
      parent node id are printed, the lattern between parentheses.  */
  friend std::ostream& operator<<(std::ostream& out, graph_t& g) {
    for(graph_t::iterator i = g.begin(), e = g.end(); i != e; ++i) {
      node_t& node = *i;
      out << node.id() << ": " << node.distance();
      if (node.parent_node())
	out << " (" << node.parent_node()->id() << ")";
      out << endl;
    }
    return out;
  }

 public:
  /** Add up the distances of the nodes, just for the sake of
      validating implementations.  */
  distance_t sum_dist() const {
    distance_t sum = 0;
    for(graph_t::const_iterator i = begin(), e = end(); i != e; ++i) {
      const node_t& node = *i;
      sum += node.distance();
    }
    return sum;
  }
};

#endif
