// $Id: WithArrayAndRange.java,v 1.2 1999/09/20 11:19:53 oliva Exp $

/* Copyright 1999 Alexandre Oliva <oliva@lsd.ic.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package BR.unicamp.Guarana.SequentialComposerAlgorithms;

import BR.unicamp.Guarana.Composer;
import BR.unicamp.Guarana.Message;
import BR.unicamp.Guarana.MetaException;
import BR.unicamp.Guarana.MetaObject;
import BR.unicamp.Guarana.Operation;
import BR.unicamp.Guarana.OperationFactory;
import BR.unicamp.Guarana.Result;
import java.util.Enumeration;

/** Implements algorithms for Composers like SequentialComposer, that
    sequentially request a fixed set of MetaObjects to handle
    Operations, Results and Messages, as well as reconfiguration,
    configuration, initialize and release requests.  It assumes the
    array of metaObjects given to each of its Operations is constant.

    <p>For each method of MetaObject, this class implements a method
    that takes the additional arguments <tt>metaObjects</tt>,
    <tt>begin</tt>, <tt>increment</tt> and <tt>end</tt>.  The first is
    the array of MetaObjects to which it should delegate the requests.
    The first MetaObject that gets the Operation is the one whose
    index is <tt>begin</tt>.  Then, <tt>increment</tt> is added to
    <tt>begin</tt> to compute the index of the next element of the
    array, and so on, until the computed index is exactly equal to
    <tt>end</tt>.  The element whose index is <tt>end</tt> is not
    processed, but the one whose index is <tt>begin</tt> is, unless it
    is equal to <tt>end</tt>, in which case no element is processed.

    <p>All algorithms are final, to avoid exposing the metaObjects
    array of its main subclass, SequentialComposer.  This could have
    been avoided differently (with final implementations of these
    methods in class SequentialComposer itself), but it probably
    wasn't worth the trouble, since there's little point in
    specializing these methods.  They used to be static within
    SequentialComposer, after all.

    @see SequentialComposer.

    @since Guaran 1.7

    @author Alexandre Oliva
    @version $Revision: 1.2 $ */
public abstract class WithArrayAndRange extends Composer {
  /** Creates an Enumeration that iterates on the selected MetaObjects
      of the given array.

      @param metaObjects an array of MetaObjects.
      
      @param begin the index of the first element of the array to be
      processed.  This is usually 0.

      @param increment the value added to an index to compute the
      index of the next element to be processed.  This is usually 1.

      @param end the index of the first element of the array that
      should not be processed.  This is usually
      <tt>metaObjects.length</tt>.
      
      @return the Enumeration.  */
  final protected Enumeration
  getMetaObjects(final MetaObject[] metaObjects,
		 final int begin,
		 final int increment,
		 final int end) {
    return new Enumeration() {
      private int i = begin;
      public boolean hasMoreElements() {
	return i != end;
      }
      public Object nextElement() {
	final MetaObject mo = metaObjects[i];
	i += increment;
	return mo;
      }
    };
  }

  /** Asks MetaObjects in the given array to handle the Operation.  If
      a MetaObject provides a Result for the Operation (as opposed to
      a request to be presented a Result or null), the previous
      MetaObjects are asked to handle the Result, through the
      invocation of the method to handle results (so even the ones
      that asked for no further notice will get the Result).  The last
      produced Result is returned.

      @see #handleResult

      @param operation the Operation to be delegated to MetaObjects.

      @param object the target Object of the Operation.

      @param metaObjects an array of MetaObjects.
      
      @param begin the index of the first element of the array to be
      processed.  This is usually 0.

      @param increment the value added to an index to compute the
      index of the next element to be processed.  This is usually 1.

      @param end the index of the first element of the array that
      should not be processed.  This is usually
      <tt>metaObjects.length</tt>.
      
      @return a Result for the Operation, if one was produced by any
      MetaObject, or a replacement Operation, if one was produced by
      any MetaObject, or a Result request, if any MetaObject asked to
      read or modify the Result.  If both read and modify requests
      were made, a modify request is returned.  */
  final protected Result
  handleOperation(final Operation operation,
		  final Object object,
		  final MetaObject[] metaObjects,
		  final int begin,
		  final int increment,
		  final int end) {
    int finmode = Result.noResultMode;
    Operation op = operation;
    for(int i = begin; i != end; i += increment) {
      Result res = null;
      try {
	res = metaObjects[i].handle(op, object);
      } catch (Throwable t) {
	res = Result.throwObject(new MetaException(t), op);
      }
      if (res != null) {
	Operation nop = res.getOperation();
	if (nop != null && nop != op) {
	    if (nop.replaced(op))
	      op = nop;
	    else /* if result is unrelated with the requested
		  * operation, ignore it */
	      continue;
	}
	final int mode = res.getMode();
	if ((mode & Result.resultValueMask) != 0)
	  return handleResult(res, object,
			      metaObjects,
			      i-increment,
			      -increment,
			      begin-increment);
	final int newmode = mode & Result.resultRequestMask;
	if (newmode > finmode)
	  finmode = newmode;
      }
    }
    if (op == operation && finmode == Result.noResultMode)
      return null;
    else
      return Result.operation(op, finmode);
  }

  /** Asks MetaObject in the given array to handle the Result.  If any
      of the MetaObjects returns a valid non-null Result, the returned
      Result replaces the one provided, even if the MetaObject had not
      requested to modify it.

      @param res the Result to be presented to the MetaObjects.

      @param object the target Object of the Operation the Result
      refers to.

      @param metaObjects an array of MetaObjects.
      
      @param begin the index of the first element of the array to be
      processed.  This is usually <tt>metaObjects.length-1<tt>.

      @param increment the value added to an index to compute the
      index of the next element to be processed.  This is usually -1.

      @param end the index of the first element of the array that
      should not be processed.  This is usually -1.
      
      @return the last Result returned by a MetaObject, or the original
      Result, if every MetaObject returned null.  */
  final protected Result
  handleResult(Result res,
	       final Object object,
	       final MetaObject[] metaObjects,
	       final int begin,
	       final int increment,
	       final int end) {
    for(int i = begin; i != end; i += increment) {
      Result nres;
      try {
	nres = metaObjects[i].handle(res, object);
      } catch (Throwable t) {
	nres = Result.throwObject(new MetaException(t),
				  res.getOperation());
      }
      if (nres != null &&
	  (nres.getMode() & Result.resultValueMask) != 0) {
	final Operation op = nres.getOperation();
	if (op != null && op.replaced(res.getOperation()))
	  res = nres;
      }
    }
    return res;
  }

  /** Delegates the Message to the selected metaObjects of the array.
      The first MetaObject that gets the Operation is the one whose
      index is <tt>begin</tt>.  Then, <tt>increment</tt> is added to
      <tt>begin</tt> to compute the index of the next element of the
      array, and so on, until the computed index is exactly equal to
      <tt>end</tt>.

      @param message the Message to be broadcasted.

      @param object the Object the Message refers to.

      @param metaObjects an array of MetaObjects.
      
      @param begin the index of the first element of the array to be
      processed.  This is usually 0.

      @param increment the value added to an index to compute the
      index of the next element to be processed.  This is usually 1.

      @param end the index of the first element of the array that
      should not be processed.  This is usually
      <tt>metaObjects.length</tt>.  */
  final protected void
  handleMessage(final Message message,
		final Object object,
		final MetaObject[] metaObjects,
		final int begin,
		final int increment,
		final int end) {
    for(int i = begin; i != end; i += increment)
      metaObjects[i].handle(message, object);
  }

  /** Asks each MetaObject to provide a MetaObject to occupy its place
      in a new Object's meta-configuration.  If all of them return
      themselves, the metaObjects argument is returned.  If all of
      them return null, return null too.  In any other case, a new,
      packed array is created, containing the non-null MetaObjects
      returned by the elements of the metaObjects array, and this new
      array is returned.

      @param newObject the Object MetaObjects should be created for.

      @param object the Object whose meta-configuration is being asked
      to create the newObject's configuration.

      @param metaObjects an array of MetaObjects.
      
      @param begin the index of the first element of the array to be
      processed.  This is usually 0.

      @param increment the value added to an index to compute the
      index of the next element to be processed.  This is usually 1.

      @param end the index of the first element of the array that
      should not be processed.  This is usually
      <tt>metaObjects.length</tt>.
      
      @return a packed array of MetaObjects with the MetaObjects
      returned by the ones in the metaObjects array, or null, if all
      of them returned null.  */
  final protected MetaObject[]
  configureArray(final Object newObject,
		 final Object object,
		 final MetaObject[] metaObjects,
		 final int begin,
		 final int increment,
		 final int end) {
    boolean same = (begin == 0
		    && end == metaObjects.length
		    && increment == 1);
    int count = 0;
    MetaObject[] newMetaObjects = null;
    for(int i = begin; i != end; i += increment) {
      final MetaObject orig = metaObjects[i];
      final MetaObject mo = orig.configure(newObject, object);
      if (mo == null)
	same = false;
      else if (same && mo == orig)
	count++;
      else {
	if (newMetaObjects == null) {
	  same = false;
	  newMetaObjects = new MetaObject[count + (end-i)/increment];
	  if (count > 0) {
	    if (increment == 1)
	      System.arraycopy(metaObjects, begin,
			       newMetaObjects, 0, count);
	    else
	      for(int j = begin, ct = 0; j != i; j += increment)
		newMetaObjects[ct++] = metaObjects[j];
	  }
	}
	newMetaObjects[count++] = mo;
      }
    }

    if (same)
      return metaObjects;

    if (count == 0)
      return null;

    if (count == newMetaObjects.length)
      return newMetaObjects;

    final MetaObject[] packedMetaObjects = new MetaObject[count];
    System.arraycopy(newMetaObjects, 0, packedMetaObjects, 0, count);
    return packedMetaObjects;
  }

  /** Delegates reconfiguration requests to all metaObjects.  If all
      of them return themselves, the metaObjects argument is returned.
      If all of them return null, return null too.  In any other case,
      a new, packed array is created, containing the non-null
      MetaObjects returned by the elements of the metaObjects array,
      and this new array is returned.

      @param object the Object whose meta-configuration should be
      affected.

      @param oldMetaObject that MetaObject that may be replaced.

      @param newMetaObject the candidate MetaObject to replace it.

      @param metaObjects an array of MetaObjects.
      
      @param begin the index of the first element of the array to be
      processed.  This is usually 0.

      @param increment the value added to an index to compute the
      index of the next element to be processed.  This is usually 1.

      @param end the index of the first element of the array that
      should not be processed.  This is usually
      <tt>metaObjects.length</tt>.
      
      @return the given array of metaObjects, if no changes were
      performed, or a new, packed array, if reconfiguration is to take
      place.

      @since Guaran 1.7  */
  final protected MetaObject[]
  reconfigureArray(final Object object,
		   final MetaObject oldMetaObject,
		   final MetaObject newMetaObject,
		   MetaObject[] metaObjects,
		   final int begin,
		   final int increment,
		   final int end) {
    boolean same = (begin == 0
		    && end == metaObjects.length
		    && increment == 1);
    int count = 0;
    MetaObject[] newMetaObjects = null;
    for(int i = begin; i != end; i += increment) {
      final MetaObject orig = metaObjects[i];
      final MetaObject
	mo = orig.reconfigure(object, oldMetaObject, newMetaObject);
      if (mo == null)
	same = false;
      else if (same && mo == orig)
	count++;
      else {
	if (newMetaObjects == null) {
	  same = false;
	  newMetaObjects = new MetaObject[count + (end-i)/increment];
	  if (count > 0) {
	    if (increment == 1)
	      System.arraycopy(metaObjects, begin,
			       newMetaObjects, 0, count);
	    else
	      for(int j = begin, ct = 0; j != i; j += increment)
		newMetaObjects[ct++] = metaObjects[j];
	  }
	}
	newMetaObjects[count++] = mo;
      }
    }

    if (same)
      return metaObjects;

    if (count == 0)
      return null;

    if (count == newMetaObjects.length)
      return newMetaObjects;

    final MetaObject[] packedMetaObjects = new MetaObject[count];
    System.arraycopy(newMetaObjects, 0, packedMetaObjects, 0, count);
    return packedMetaObjects;
  }

  /** Delegates the initialization to all metaObjects.

      @param factory the Operation Factory to be used to create
      Operations for that Object.

      @param object the Object MetaObjects should become able to handle.

      @param metaObjects an array of MetaObjects.
      
      @param begin the index of the first element of the array to be
      processed.  This is usually 0.

      @param increment the value added to an index to compute the
      index of the next element to be processed.  This is usually 1.

      @param end the index of the first element of the array that
      should not be processed.  This is usually
      <tt>metaObjects.length</tt>.  */
  final protected void
  initialize(final OperationFactory factory,
	     final Object object,
	     final MetaObject[] metaObjects,
	     final int begin,
	     final int increment,
	     final int end) {
    for(int i = begin; i != end; i += increment) {
      final MetaObject mo = metaObjects[i];
      mo.initialize(factory, object);
    }
  }

  /** Delegates the release information to all metaObjects.

      @param object the Object that should no longer be handled.

      @param metaObjects an array of MetaObjects.
      
      @param begin the index of the first element of the array to be
      processed.  This is usually <tt>metaObjects.length-1<tt>.

      @param increment the value added to an index to compute the
      index of the next element to be processed.  This is usually -1.

      @param end the index of the first element of the array that
      should not be processed.  This is usually -1.  */
  final protected void
  release(final Object object,
	  final MetaObject[] metaObjects,
	  final int begin,
	  final int increment,
	  final int end) {
    for(int i = begin; i != end; i += increment)
      metaObjects[i].release(object);
  }
}
