// $Id: SequentialComposer.java,v 1.12 1999/09/16 12:31:08 oliva Exp $

/* Copyright 1997,1998,1999 Alexandre Oliva <oliva@lsd.ic.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package BR.unicamp.Guarana;

import java.util.Enumeration;
import BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray;

/** Implements a Composer that sequentially requests a fixed set of
    MetaObjects to handle Operations, Results and Messages, as well as
    reconfiguration, configuration, initialize and release requests.
    It assumes the array of metaObjects is constant.  Reconfiguration
    is only accomplished with the creation of a new composer.

    <p>The most important adaptation point for subclasses is the
    method newComposer; if it is not overridden, configure and
    reconfigure requests may cause the creation of SequentialComposers
    instead of instances of the subclass, whenever a configuration or
    reconfiguration results in a different array of MetaObjects.
    Furthermore, if configure or reconfigure result a unitary or empty
    array of MetaObjects, SequentialComposers will leave, being
    replaced with null or with the only MetaObject in the array.
    These policies can be changed by overriding composerForArray, that
    affects both configure and reconfigure, or composerForConfigure
    and/or composerForReconfigure.

    @see #newComposer
    @see #composerForArrayWithPolicies
    @see BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray#composerForArray
    @see BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray#composerForConfigure
    @see BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray#composerForReconfigure

    @author Alexandre Oliva
    @version $Revision: 1.12 $  */
public class SequentialComposer extends WithArray {
  /** Contains the metaObjects this composer delegates to.  The array
      must never be modified.  */
  final private MetaObject[] metaObjects;

  /** Creates a Composer that will sequentially delegate operations
      and messages to the elements of metaObjects.  The array is
      cloned, so as to ensure that it is never changed.

      @param metaObjects the set of MetaObjects the Composer should
      delegate to.  */
  public
  SequentialComposer(final MetaObject[] metaObjects) {
    this.metaObjects = (MetaObject[])metaObjects.clone();
  }

  /** Obtains the MetaObjects this Composer delegates to.

      @return a clone of the internal array of MetaObjects.  */
  public MetaObject[]
  getMetaObjectsArray() {
    return (MetaObject[])metaObjects.clone();
  }

  /** Invokes the corresponding method that takes an array of
      MetaObjects.

      @see BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray#handleOperation  */
  public Result
  handle(final Operation operation,
	 final Object object) {
    return handleOperation(operation, object, metaObjects);
  }

  /** Invokes the corresponding method that takes an array of
      MetaObjects.

      @see BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray#getMetaObjects  */
  public Enumeration
  getMetaObjects() {
    return getMetaObjects(metaObjects);
  }

  /** Invokes the corresponding method that takes an array of
      MetaObjects.

      @see BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray#handleResult  */
  public Result
  handle(final Result result,
	 final Object object) {
    return handleResult(result, object, metaObjects);
  }

  /** Invokes the corresponding method that takes an array of
      MetaObjects.

      @see BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray#handleMessage  */
  public void
  handle(final Message message,
	 final Object object) {
    handleMessage(message, object, metaObjects);
  }

  /** Invokes the corresponding method that takes an array of
      MetaObjects.

      @see BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray#configure  */
  public MetaObject
  configure(final Object newObject,
	    final Object object) {
    return configure(newObject, object, metaObjects);
  }

  /** Invokes the corresponding method that takes an array of
      MetaObjects.

      @see BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray#reconfigure  */
  public synchronized MetaObject
  reconfigure(final Object object,
	      final MetaObject oldMetaObject,
	      final MetaObject newMetaObject) {
    return reconfigure(object, oldMetaObject, newMetaObject,
		       metaObjects);
  }

  /** Invoke the corresponding method that takes an array of
      MetaObjects.

      @see BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray#initialize  */
  public void
  initialize(final OperationFactory factory,
	     final Object object) {
    initialize(factory, object, metaObjects);
  }

  /** Invoke the corresponding method that takes an array of
      MetaObjects.

      @see BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray#release  */
  public void
  release(final Object object) {
    release(object, metaObjects);
  }

  /** If may_propagate_itself is true, compare the given metaObjects
      array with the internal array.  If they're the same array,
      return itself.  Otherwise, delegate the invocation to the
      superclass.

      @see BR.unicamp.Guarana.SequentialComposerAlgorithms.WithArray#composerForArrayWithPolicies
  */
  protected MetaObject
  composerForArrayWithPolicies(final MetaObject[] metaObjects,
			       final boolean may_propagate_itself,
			       final boolean may_leave_when_empty,
			       final boolean may_leave_when_unitary) {
    if (may_propagate_itself && this.metaObjects == metaObjects)
      return this;

    return super.composerForArrayWithPolicies(metaObjects,
					      may_propagate_itself,
					      may_leave_when_empty,
					      may_leave_when_unitary);
  }

  /** Creates a new SequentialComposer with the given array of
      metaObjects.  It may be indirectly called from configure and
      reconfigure, when a new composer must be created.  A subclass
      should override this method to create an instance of the
      subclass.

      @param metaObjects the set of MetaObjects the new Composer
      should delegate to.

      @return the new Composer.

      @since Guaran 1.7  */
  protected MetaObject
  newComposer(final MetaObject[] metaObjects) {
    return new SequentialComposer(metaObjects);
  }
}
