/*
 * GuaraNative.c
 * Utility functions for native implementation of Guarana
 *
 * Copyright 1997,1998,1999,2000 Alexandre Oliva <oliva@lsd.ic.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "GuaranaIntrn.h"

#include "baseClasses.h"
#include "thread.h"
#include "stringSupport.h"

#include "debug.h"

char * guarana_version = GUARANA_VERSION;

/* These are initialized by kaffe_guarana_init_function_pointers, in Class.c */

struct Hjava_lang_reflect_Constructor* (*p_makeConstructor)(struct Hjava_lang_Class* clazz, int slot);

struct Hjava_lang_reflect_Method* (*p_makeMethod)(struct Hjava_lang_Class* clazz, int slot);

struct Hjava_lang_reflect_Field* (*p_makeField)(struct Hjava_lang_Class* clazz, int slot);

Hjava_lang_String* (*p_getClassName)(struct Hjava_lang_Class *clazz);

static Hjava_lang_Class *guaranaClass = 0, *factoryClass = 0,
  *operationClass = 0, *resultClass = 0, *newObjMsgClass;
static Method *setMetaObjectOf = 0, *newObjConstr = 0,
  *broadcast = 0;
static jobject *Guarana_blocker = 0;

NativeOperation *guarana_new_native_operation(struct Hjava_lang_Object *object, struct HBR_unicamp_Guarana_Operation *replaced) {
  NativeOperation *op;
  if (!operationClass) {
    errorInfo info;
    operationClass = lookupClass("BR/unicamp/Guarana/Operation", NULL, &info);
    if (!operationClass)
      throwError(&info);
    CLASS_FSIZE(operationClass) = sizeof(NativeOperation);
  }

  op = (NativeOperation*)gc_malloc(sizeof(NativeOperation), GC_ALLOC_OPERATION);
  op->base.dtable = operationClass->dtable;
  op->base.meta_object = 0;

DBG(NEWOBJECT,
	dprintf("newOperation %x\n", op);
    )

  op->object = object;
  op->replaced = replaced;
  op->thread = replaced ? ((NativeOperation*)replaced)->thread : 0;
  op->kind = 0;
  return op;
}

struct HBR_unicamp_Guarana_Result *guarana_new_result_object(int /* enum result_type */ kind, struct HBR_unicamp_Guarana_Operation *op, jvalue result) {
  NativeResult *res;
  errorInfo info;
  if (!resultClass) {
    resultClass = lookupClass("BR/unicamp/Guarana/Result", NULL, &info);
    if (!resultClass)
      throwError(&info);
    CLASS_FSIZE(resultClass) = sizeof(NativeResult);
  }
  res = (NativeResult*)newObjectNoMessage(resultClass, &info);
  if (!res)
      throwError(&info);
  
  res->kind = kind;
  res->op = op;
  res->res = result;
  return (struct HBR_unicamp_Guarana_Result *)res;
}

OperationFactory *guarana_new_operation_factory(struct Hjava_lang_Object *object, struct HBR_unicamp_Guarana_MetaObject *metaObject) {
  OperationFactory *fact;
  errorInfo info;
  if (!factoryClass) {
    factoryClass = lookupClass("BR/unicamp/Guarana/OperationFactoryInternal",
			       NULL, &info);
    if (!factoryClass)
      throwError(&info);
    CLASS_FSIZE(factoryClass) = sizeof(OperationFactory);
  }
  fact = (OperationFactory*)newObjectNoMessage(factoryClass, &info);
  if (!fact)
    throwError(&info);
  fact->obj = object;
  fact->mobj = metaObject;
  return fact;
}

struct Hjava_lang_Object *guarana_wrap(struct Hjava_lang_Class *clazz, jvalue val, int *size) {
  int s;
  if (!size)
    size = &s;
  if (!CLASS_IS_PRIMITIVE(clazz)) {
    *size = 1;
    return val.l;
  }
  switch (CLASS_PRIM_SIG(clazz)) {
  case 'Z':
    *size = 1;
    return execute_java_constructor(0, 0, javaLangBooleanClass, "(Z)V", val.i);
  case 'B':
    *size = 1;
    return execute_java_constructor(0, 0, javaLangByteClass, "(B)V", val.i);
  case 'S':
    *size = 1;
    return execute_java_constructor(0, 0, javaLangShortClass, "(S)V", val.i);
  case 'I':
    *size = 1;
    return execute_java_constructor(0, 0, javaLangIntegerClass, "(I)V", val.i);
  case 'J':
    *size = 2;
    return execute_java_constructor(0, 0, javaLangLongClass, "(J)V", val.j);
  case 'C':
    *size = 1;
    return execute_java_constructor(0, 0, javaLangCharacterClass, "(C)V", val.i);
  case 'F':
    *size = 1;
    return execute_java_constructor(0, 0, javaLangFloatClass, "(F)V", val.f);
  case 'D':
    *size = 2;
    return execute_java_constructor(0, 0, javaLangDoubleClass, "(D)V", val.d);
  case 'V':
    *size = 0;
    return 0;
  default:
    ABORT();
  }
  return 0; /* never reached */
}

int guarana_unwrap(struct Hjava_lang_Class *clazz, jvalue *val, struct Hjava_lang_Object *wrapper) {
  if (!CLASS_IS_PRIMITIVE(clazz)) {
    if (wrapper && !instanceof(clazz, OBJECT_CLASS(wrapper)))
      SignalError("java/lang/IllegalArgumentException", "object is not an instance of the requested class");
    val->l = wrapper;
    return 1;
  } else switch (CLASS_PRIM_SIG(clazz)) {
  case 'Z':
    if (OBJECT_CLASS(wrapper) != javaLangBooleanClass)
      SignalError("java/lang/IllegalArgumentException", "incorrect wrapper class");
    val->i = do_execute_java_method(wrapper, "booleanValue", "()Z", 0, 0).i;
    return 1;
  case 'B':
    if (OBJECT_CLASS(wrapper) != javaLangByteClass)
      SignalError("java/lang/IllegalArgumentException", "incorrect wrapper class");
    val->i = do_execute_java_method(wrapper, "byteValue", "()B", 0, 0).i;
    return 1;
  case 'S':
    if (OBJECT_CLASS(wrapper) != javaLangShortClass)
      SignalError("java/lang/IllegalArgumentException", "incorrect wrapper class");
    val->i = do_execute_java_method(wrapper, "shortValue", "()S", 0, 0).i;
    return 1;
  case 'I':
    if (OBJECT_CLASS(wrapper) != javaLangIntegerClass)
      SignalError("java/lang/IllegalArgumentException", "incorrect wrapper class");
    val->i = do_execute_java_method(wrapper, "intValue", "()I", 0, 0).i;
    return 1;
  case 'J':
    if (OBJECT_CLASS(wrapper) != javaLangLongClass)
      SignalError("java/lang/IllegalArgumentException", "incorrect wrapper class");
    val->j = do_execute_java_method(wrapper, "longValue", "()J", 0, 0).j;
    return 2;
  case 'C':
    if (OBJECT_CLASS(wrapper) != javaLangCharacterClass)
      SignalError("java/lang/IllegalArgumentException", "incorrect wrapper class");
    val->i = do_execute_java_method(wrapper, "charValue", "()C", 0, 0).i;
    return 1;
  case 'F':
    if (OBJECT_CLASS(wrapper) != javaLangFloatClass)
      SignalError("java/lang/IllegalArgumentException", "incorrect wrapper class");
    val->f = do_execute_java_method(wrapper, "floatValue", "()F", 0, 0).f;
    return 1;
  case 'D':
    if (OBJECT_CLASS(wrapper) != javaLangDoubleClass)
      SignalError("java/lang/IllegalArgumentException", "incorrect wrapper class");
    val->d = do_execute_java_method(wrapper, "doubleValue", "()D", 0, 0).d;
    return 2;
  case 'V':
    val->l = 0;
    return 0;
  default:
    ABORT();
  }
  return 0; /* never reached */
}

NativeResult *guarana_handle_operation(NativeOperation *op) {
  NativeResult *res;

  res = (NativeResult*)(do_execute_java_class_method("BR/unicamp/Guarana/Guarana", NULL, "perform", "(LBR/unicamp/Guarana/Operation;)LBR/unicamp/Guarana/Result;", op).l);

  if (res->kind & res_thrown)
    throwException(res->res.l);

  return res;
}

void
guarana_slots_reify_method_invocation(struct Hjava_lang_Object *obj, Method *method, slots *args, slots *retval) {
  struct HBR_unicamp_Guarana_MetaObject *mo;
  if ((method->accflags & ACC_STATIC) != 0)
    obj = &method->class->head;
  mo = obj->meta_object;
  if (mo) {
    NativeOperation *op;
    NativeResult *res;
    const char *sig;
    int argcount;

    op = guarana_new_native_operation(obj, 0);
    if (utf8ConstEqual(method->name, init_name) ||
	utf8ConstEqual(method->name, constructor_name))
      op->kind |= op_constructor_invocation;
    else
      op->kind |= op_method_invocation;
    op->op_id.method = method;
    sig = METHOD_SIGD(method);
    argcount = sizeofSig(&sig, true);
    op->op_arg.l = argcount ? ALLOC_ARGVEC_FOR(argcount) : 0;
    if (argcount)
	memcpy(op->op_arg.l, args, argcount * sizeof(jvalue));

    res = guarana_handle_operation(op);

    switch(*METHOD_RET_TYPE(method)) {
    case 'Z':
    case 'B':
    case 'S':
    case 'C':
    case 'I':
      retval->v.tint = res->res.i;
      break;
    case 'J':
      retval->v.tlong = res->res.j;
      break;
    case 'F':
      retval->v.tfloat = res->res.f;
      break;
    case 'D':
      retval->v.tdouble = res->res.d;
      break;
    case 'L':
    case '[':
      retval->v.taddr = res->res.l;
      break;
    case 'V':
      break;
    default:
      ABORT();
    }
  } else {
    callMethodA(method, METHOD_INDIRECTMETHOD(method), obj, (jvalue*)args, (jvalue*)retval, 1);
  }
};

void
guarana_slots_resolve_method_invocation(struct Hjava_lang_Object *obj, Method *method, slots *args, slots *retval) {
  if (!METHOD_IS_STATIC(method))
#ifdef DMETHOD_DESC
      method = DMETHOD_DESC(obj->dtable->method[method->idx]);
#else
      method = DMETHOD_ADDR(obj->dtable->method[method->idx]);
#endif
  guarana_slots_reify_method_invocation(obj, method, args, retval);
}

void jvoid_reify_method_invocation(Method *method, Hjava_lang_Object *obj, ...) {
  va_list arglist;
  jbool isStatic = (method->accflags & ACC_STATIC) != 0;

  va_start(arglist, obj);
  do_execute_java_method_v(isStatic ? 0 : obj, 0, 0, method,
			   isStatic, arglist);
  va_end(arglist);
}

jref jref_reify_method_invocation(Method *method, Hjava_lang_Object *obj, ...) {
  va_list arglist;
  jbool isStatic = (method->accflags & ACC_STATIC) != 0;
  jvalue result;

  va_start(arglist, obj);
  result = do_execute_java_method_v(isStatic ? 0 : obj, 0, 0, method,
				    isStatic, arglist);
  va_end(arglist);

  return result.l;
}

jword jboolean_reify_method_invocation(Method *method, Hjava_lang_Object *obj, ...) {
  va_list arglist;
  jbool isStatic = (method->accflags & ACC_STATIC) != 0;
  jvalue result;

  va_start(arglist, obj);
  result = do_execute_java_method_v(isStatic ? 0 : obj, 0, 0, method,
				    isStatic, arglist);
  va_end(arglist);

  return result.i;
}

jword jchar_reify_method_invocation(Method *method, Hjava_lang_Object *obj, ...) {
  va_list arglist;
  jbool isStatic = (method->accflags & ACC_STATIC) != 0;
  jvalue result;

  va_start(arglist, obj);
  result = do_execute_java_method_v(isStatic ? 0 : obj, 0, 0, method,
				    isStatic, arglist);
  va_end(arglist);

  return result.i;
}

jword jbyte_reify_method_invocation(Method *method, Hjava_lang_Object *obj, ...) {
  va_list arglist;
  jbool isStatic = (method->accflags & ACC_STATIC) != 0;
  jvalue result;

  va_start(arglist, obj);
  result = do_execute_java_method_v(isStatic ? 0 : obj, 0, 0, method,
				    isStatic, arglist);
  va_end(arglist);

  return result.i;
}

jword jshort_reify_method_invocation(Method *method, Hjava_lang_Object *obj, ...) {
  va_list arglist;
  jbool isStatic = (method->accflags & ACC_STATIC) != 0;
  jvalue result;

  va_start(arglist, obj);
  result = do_execute_java_method_v(isStatic ? 0 : obj, 0, 0, method,
				    isStatic, arglist);
  va_end(arglist);

  return result.i;
}

jword jint_reify_method_invocation(Method *method, Hjava_lang_Object *obj, ...) {
  va_list arglist;
  jbool isStatic = (method->accflags & ACC_STATIC) != 0;
  jvalue result;

  va_start(arglist, obj);
  result = do_execute_java_method_v(isStatic ? 0 : obj, 0, 0, method,
				    isStatic, arglist);
  va_end(arglist);

  return result.i;
}

jlong jlong_reify_method_invocation(Method *method, Hjava_lang_Object *obj, ...) {
  va_list arglist;
  jbool isStatic = (method->accflags & ACC_STATIC) != 0;
  jvalue result;

  va_start(arglist, obj);
  result = do_execute_java_method_v(isStatic ? 0 : obj, 0, 0, method,
				    isStatic, arglist);
  va_end(arglist);

  return result.j;
}

jfloat jfloat_reify_method_invocation(Method *method, Hjava_lang_Object *obj, ...) {
  va_list arglist;
  jbool isStatic = (method->accflags & ACC_STATIC) != 0;
  jvalue result;

  va_start(arglist, obj);
  result = do_execute_java_method_v(isStatic ? 0 : obj, 0, 0, method,
				    isStatic, arglist);
  va_end(arglist);

  return result.f;
}

jdouble jdouble_reify_method_invocation(Method *method, Hjava_lang_Object *obj, ...) {
  va_list arglist;
  jbool isStatic = (method->accflags & ACC_STATIC) != 0;
  jvalue result;

  va_start(arglist, obj);
  result = do_execute_java_method_v(isStatic ? 0 : obj, 0, 0, method,
				    isStatic, arglist);
  va_end(arglist);

  return result.d;
}

void
guarana_va_list_reify_method_invocation(struct Hjava_lang_Object *obj, Method *method, va_list args, slots *retval) {
  NativeOperation *op;
  NativeResult *res;
  const char *sig;
  int argcount, i, len;
  jvalue *argl;

  if ((method->accflags & ACC_STATIC) != 0)
    obj = &method->class->head;
  op = guarana_new_native_operation(obj, 0);
  if (utf8ConstEqual(method->name, init_name) ||
      utf8ConstEqual(method->name, constructor_name))
    op->kind |= op_constructor_invocation;
  else
    op->kind |= op_method_invocation;
  op->op_id.method = method;
  sig = METHOD_SIGD(method);
  argcount = sizeofSig(&sig, true);
  op->op_arg.l = argcount ? ALLOC_ARGVEC_FOR(argcount) : 0;
  i = 0;
  len = METHOD_NARGS(method);
  
  for(argl = (jvalue*)op->op_arg.l; i < len; ++i, ++argl) {
    switch (*METHOD_ARG_TYPE(method, i)) {
    case 'Z':
    case 'B':
    case 'S':
    case 'C':
    case 'I':
      argl->i = va_arg(args, jint);
      break;
    case 'J':
      argl->j = va_arg(args, jlong);
      ++argl;
      break;
    case 'F':
      argl->f = (jfloat)va_arg(args, jdouble);
      break;
    case 'D':
      argl->d = va_arg(args, jdouble);
      ++argl;
      break;
    case '[':
    case 'L':
      argl->l = va_arg(args, jref);
      break;
    default:
      ABORT();
    }
  }
    
  res = guarana_handle_operation(op);

  /* skip to return type */

  switch (*METHOD_RET_TYPE(method)) {
    case 'Z':
    case 'B':
    case 'S':
    case 'C':
    case 'I':
      retval->v.tint = res->res.i;
      break;
    case 'J':
      retval->v.tlong = res->res.j;
      break;
    case 'F':
      retval->v.tfloat = res->res.f;
      break;
    case 'D':
      retval->v.tdouble = res->res.d;
      break;
    case 'L':
    case '[':
      retval->v.taddr = res->res.l;
      break;
    case 'V':
      break;
    default:
      ABORT();
  }
}

void
guarana_va_list_resolve_method_invocation(struct Hjava_lang_Object *obj, Method *method, va_list args, slots *retval) {
  if (!METHOD_IS_STATIC(method))
#ifdef DMETHOD_DESC
      method = DMETHOD_DESC(obj->dtable->method[method->idx]);
#else
      method = DMETHOD_ADDR(obj->dtable->method[method->idx]);
#endif
  guarana_va_list_reify_method_invocation(obj, method, args, retval);
}

void
guarana_reify_field_load(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field, jvalue *wrap) {
  NativeOperation *op;
  NativeResult *res;

  op = guarana_new_native_operation(o, 0);
  op->kind |= op_field_read;
  op->op_id.field_id.clazz = clazz;
  op->op_id.field_id.field = field;

  res = guarana_handle_operation(op);

  *wrap = res->res;
}

jref
jref_reify_field_load(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field) {
  jvalue wrap;
  guarana_reify_field_load(o, clazz, field, &wrap);
  return wrap.l;
}

jword
jboolean_reify_field_load(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field) {
  jvalue wrap;
  guarana_reify_field_load(o, clazz, field, &wrap);
  return wrap.i;
}

jword
jchar_reify_field_load(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field) {
  jvalue wrap;
  guarana_reify_field_load(o, clazz, field, &wrap);
  return wrap.i;
}

jword
jbyte_reify_field_load(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field) {
  jvalue wrap;
  guarana_reify_field_load(o, clazz, field, &wrap);
  return wrap.i;
}

jword
jshort_reify_field_load(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field) {
  jvalue wrap;
  guarana_reify_field_load(o, clazz, field, &wrap);
  return wrap.i;
}

jword
jint_reify_field_load(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field) {
  jvalue wrap;
  guarana_reify_field_load(o, clazz, field, &wrap);
  return wrap.i;
}

jlong
jlong_reify_field_load(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field) {
  jvalue wrap;
  guarana_reify_field_load(o, clazz, field, &wrap);
  return wrap.j;
}

jfloat
jfloat_reify_field_load(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field) {
  jvalue wrap;
  guarana_reify_field_load(o, clazz, field, &wrap);
  return wrap.f;
}

jdouble
jdouble_reify_field_load(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field) {
  jvalue wrap;
  guarana_reify_field_load(o, clazz, field, &wrap);
  return wrap.d;
}

void
guarana_reify_field_store(struct Hjava_lang_Object *o, struct Hjava_lang_Class *clazz, Field *field, jvalue *val) {
  NativeOperation *op;

  op = guarana_new_native_operation(o, 0);
  op->kind |= op_field_write;
  op->op_id.field_id.clazz = clazz;
  op->op_id.field_id.field = field;
  op->op_arg = *val;

  guarana_handle_operation(op);
}

void
jref_reify_field_store(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field, jref value) {
  jvalue wrap;
  wrap.l = value;
  guarana_reify_field_store(o, clazz, field, &wrap);
}

void
jboolean_reify_field_store(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field, jword value) {
  jvalue wrap;
  wrap.i = value;
  guarana_reify_field_store(o, clazz, field, &wrap);
}

void
jchar_reify_field_store(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field, jword value) {
  jvalue wrap;
  wrap.i = value;
  guarana_reify_field_store(o, clazz, field, &wrap);
}

void
jbyte_reify_field_store(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field, jword value) {
  jvalue wrap;
  wrap.i = value;
  guarana_reify_field_store(o, clazz, field, &wrap);
}

void
jshort_reify_field_store(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field, jword value) {
  jvalue wrap;
  wrap.i = value;
  guarana_reify_field_store(o, clazz, field, &wrap);
}

void
jint_reify_field_store(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field, jword value) {
  jvalue wrap;
  wrap.i = value;
  guarana_reify_field_store(o, clazz, field, &wrap);
}

void
jlong_reify_field_store(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field, jlong value) {
  jvalue wrap;
  wrap.j = value;
  guarana_reify_field_store(o, clazz, field, &wrap);
}

void
jfloat_reify_field_store(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field, jfloat value) {
  jvalue wrap;
  wrap.f = value;
  guarana_reify_field_store(o, clazz, field, &wrap);
}

void
jdouble_reify_field_store(struct Hjava_lang_Object *o, Hjava_lang_Class *clazz, Field *field, jdouble value) {
  jvalue wrap;
  wrap.d = value;
  guarana_reify_field_store(o, clazz, field, &wrap);
}

void
guarana_reify_array_load(struct Hjava_lang_Object *o, int index, jvalue *wrap) {
  NativeOperation *op;
  NativeResult *res;

  op = guarana_new_native_operation(o, 0);
  op->kind |= op_array_read;
  op->op_id.index = index;

  res = guarana_handle_operation(op);

  *wrap = res->res;
}

jref
jref_reify_array_load(struct Hjava_lang_Object *o, int index) {
  jvalue wrap;
  guarana_reify_array_load(o, index, &wrap);
  return wrap.l;
}

jword
jboolean_reify_array_load(struct Hjava_lang_Object *o, int index) {
  jvalue wrap;
  guarana_reify_array_load(o, index, &wrap);
  return wrap.i;
}

jword
jchar_reify_array_load(struct Hjava_lang_Object *o, int index) {
  jvalue wrap;
  guarana_reify_array_load(o, index, &wrap);
  return wrap.i;
}

jword
jbyte_reify_array_load(struct Hjava_lang_Object *o, int index) {
  jvalue wrap;
  guarana_reify_array_load(o, index, &wrap);
  return wrap.i;
}

jword
jshort_reify_array_load(struct Hjava_lang_Object *o, int index) {
  jvalue wrap;
  guarana_reify_array_load(o, index, &wrap);
  return wrap.i;
}

jword
jint_reify_array_load(struct Hjava_lang_Object *o, int index) {
  jvalue wrap;
  guarana_reify_array_load(o, index, &wrap);
  return wrap.i;
}

jlong
jlong_reify_array_load(struct Hjava_lang_Object *o, int index) {
  jvalue wrap;
  guarana_reify_array_load(o, index, &wrap);
  return wrap.j;
}

jfloat
jfloat_reify_array_load(struct Hjava_lang_Object *o, int index) {
  jvalue wrap;
  guarana_reify_array_load(o, index, &wrap);
  return wrap.f;
}

jdouble
jdouble_reify_array_load(struct Hjava_lang_Object *o, int index) {
  jvalue wrap;
  guarana_reify_array_load(o, index, &wrap);
  return wrap.d;
}

void
guarana_reify_array_store(struct Hjava_lang_Object *o, int index, jvalue *val) {
  NativeOperation *op;

  op = guarana_new_native_operation(o, 0);
  op->kind |= op_array_write;
  op->op_id.index = index;
  op->op_arg = *val;

  guarana_handle_operation(op);
}

void
jref_reify_array_store(struct Hjava_lang_Object *o, int index, jref value) {
  jvalue wrap;
  wrap.l = value;
  guarana_reify_array_store(o, index, &wrap);
}

void
jboolean_reify_array_store(struct Hjava_lang_Object *o, int index, jword value) {
  jvalue wrap;
  wrap.i = value;
  guarana_reify_array_store(o, index, &wrap);
}

void
jchar_reify_array_store(struct Hjava_lang_Object *o, int index, jword value) {
  jvalue wrap;
  wrap.i = value;
  guarana_reify_array_store(o, index, &wrap);
}

void
jbyte_reify_array_store(struct Hjava_lang_Object *o, int index, jword value) {
  jvalue wrap;
  wrap.i = value;
  guarana_reify_array_store(o, index, &wrap);
}

void
jshort_reify_array_store(struct Hjava_lang_Object *o, int index, jword value) {
  jvalue wrap;
  wrap.i = value;
  guarana_reify_array_store(o, index, &wrap);
}

void
jint_reify_array_store(struct Hjava_lang_Object *o, int index, jword value) {
  jvalue wrap;
  wrap.i = value;
  guarana_reify_array_store(o, index, &wrap);
}

void
jlong_reify_array_store(struct Hjava_lang_Object *o, int index, jlong value) {
  jvalue wrap;
  wrap.j = value;
  guarana_reify_array_store(o, index, &wrap);
}

void
jfloat_reify_array_store(struct Hjava_lang_Object *o, int index, jfloat value) {
  jvalue wrap;
  wrap.f = value;
  guarana_reify_array_store(o, index, &wrap);
}

void
jdouble_reify_array_store(struct Hjava_lang_Object *o, int index, jdouble value) {
  jvalue wrap;
  wrap.d = value;
  guarana_reify_array_store(o, index, &wrap);
}

jint
guarana_reify_arraylength(struct Hjava_lang_Object *o) {
  struct HBR_unicamp_Guarana_MetaObject *mo = o->meta_object;
  NativeOperation *op;
  NativeResult *res;
  jvalue wrap;

  if (!mo)
    return obj_length((Array*)o);

  op = guarana_new_native_operation(o, 0);
  op->kind |= op_array_length;

  res = guarana_handle_operation(op);

  wrap = res->res;

  return wrap.i;
}

void
guarana_reify_monitorEnter(struct Hjava_lang_Object *o) {
  NativeOperation *op = guarana_new_native_operation(o, 0);
  op->kind |= op_monitor_enter;

  guarana_handle_operation(op);
}

void
guarana_reify_monitorExit(struct Hjava_lang_Object *o) {
  NativeOperation *op = guarana_new_native_operation(o, 0);
  op->kind |= op_monitor_exit;

  guarana_handle_operation(op);
}

struct Hjava_lang_Object*
guarana_meta_init(struct Hjava_lang_Object *obj, struct Hjava_lang_Class *cc, struct Hjava_lang_Object *creator, errorInfo* einfo) {
  struct HBR_unicamp_Guarana_MetaObject *mo;

  if (creator && creator->meta_object) {
    jobject newMetaObject;

    if (!guaranaClass) {
      guaranaClass = lookupClass("BR/unicamp/Guarana/Guarana", NULL, einfo);
      if (!guaranaClass)
	return 0;
    }

    if (!Guarana_blocker) {
	Field *blockerField = lookupClassField(guaranaClass, utf8ConstNew("blocker", -1), true, einfo);
	if (!blockerField)
	    return 0;
	Guarana_blocker = *(jobject*)FIELD_ADDRESS(blockerField);
	if (!Guarana_blocker) {
	  postExceptionMessage(einfo, JAVA_LANG(NullPointerException),
			       "Guarana.blocker must not be null");
	  return 0;
	}
    }

    obj->meta_object = (struct HBR_unicamp_Guarana_MetaObject*)Guarana_blocker;
    
    /* FIXME: einfo? */
    newMetaObject = do_execute_java_method(creator->meta_object, "configure", "(Ljava/lang/Object;Ljava/lang/Object;)LBR/unicamp/Guarana/MetaObject;", 0, 0, obj, creator).l;

    if (!setMetaObjectOf) {
      setMetaObjectOf = lookupClassMethod(guaranaClass, "setMetaObjectOf", "(Ljava/lang/Object;LBR/unicamp/Guarana/MetaObject;LBR/unicamp/Guarana/MetaObject;)V", einfo);
      if (!setMetaObjectOf)
	return 0;
    }
    /* FIXME: einfo? */
    do_execute_java_method(0, 0, 0, setMetaObjectOf, 1, obj, (jobject)obj->meta_object, newMetaObject);
  }

  mo = cc->head.meta_object;
  if (mo) {
    /* implicit creation of NewObject messages must not be
       reified, because this would lead to infinite recursion
       whenever a MetaObject is associated with class
       NewObject */

    struct Hjava_lang_Object *newObjMsg;

    if (!newObjMsgClass) {
      newObjMsgClass = lookupClass("BR/unicamp/Guarana/NewObject",
				   NULL, einfo);
      if (!newObjMsgClass)
	return 0;
    }

    if (!newObjConstr) {
      newObjConstr = lookupClassMethod(newObjMsgClass, constructor_name->data,
				       "(Ljava/lang/Object;)V", einfo);
      if (!newObjConstr)
	return 0;
    }

    newObjMsg = newObjectNoMessage(newObjMsgClass, einfo);
    if (!newObjMsg)
      return 0;
    /* FIXME: einfo? */
    do_execute_java_method(newObjMsg, 0, 0, newObjConstr, 0, obj);
    
    if (!guaranaClass) {
      guaranaClass = lookupClass("BR/unicamp/Guarana/Guarana", NULL, einfo);
      if (!guaranaClass)
	return 0;
    }

    if (!broadcast) {
      broadcast = lookupClassMethod(guaranaClass,
				    "broadcast",
				    "(LBR/unicamp/Guarana/Message;Ljava/lang/Object;)V",
				    einfo);
      if (!broadcast)
	return 0;
    }

    /* FIXME: einfo? */
    do_execute_java_method(0, 0, 0, broadcast, 1,
			   newObjMsg, (jobject)OBJECT_CLASS(obj));
  }

  return obj;
}  

struct Hjava_lang_Object*
guarana_newObject(struct Hjava_lang_Class *cc, struct Hjava_lang_Object *creator, errorInfo* einfo) {
  Hjava_lang_Object *obj = newObjectNoMessage(cc, einfo);
  if (!obj)
    return 0;
  return guarana_meta_init(obj, cc, creator, einfo);
}

struct Hjava_lang_Object*
newObjectChecked(struct Hjava_lang_Class *cc, errorInfo* einfo) {
  return guarana_newObject(cc, 0, einfo);
}

struct Hjava_lang_Object*
guarana_newArray(struct Hjava_lang_Class *cc, int count, struct Hjava_lang_Object *creator, errorInfo *einfo) {
  Hjava_lang_Object *array = newArrayNoMessage(cc, count, einfo);
  if (!array)
    return 0;
  return guarana_meta_init(array, cc, creator, einfo);
}

Hjava_lang_Object*
newArrayChecked(Hjava_lang_Class* eltype, int count, errorInfo* einfo)
{
  return guarana_newArray(eltype, count, (jobject)0, einfo);
}

/* guarana_newMultiArray defined in object.c */

Hjava_lang_Object*
newMultiArrayChecked(Hjava_lang_Class* clazz, int* dims, errorInfo *einfo)
{
  return guarana_newMultiArray(clazz, 0, dims, einfo);
}

/*
 * Walk an array of primitive type.
 */
void
walkPrimArray(Collector *collector, void* base, uint32 size)
{
	Hjava_lang_Object* arr;

        arr = (Hjava_lang_Object*)base;
        if (arr->dtable == 0) {                 /* see walkObject */
                return;
        }

	GC_markObject(collector, arr->meta_object);
}

/*
 * Walk an Operation object.
 */
void
walkOperation(Collector *collector, void* base, uint32 size)
{
	NativeOperation * nop = (NativeOperation*)base;

        if (nop->base.dtable == 0) {                 /* see walkObject */
                return;
        }

	GC_markObject(collector, nop->base.meta_object);

	GC_markObject(collector, nop->object);

	GC_markObject(collector, nop->thread);

	GC_markObject(collector, nop->replaced);

	switch (nop->kind & op_kind_mask) {
	case op_nop:
	    break;

	case op_method_invocation:
	case op_constructor_invocation:
	    GC_markObject(collector, nop->op_id.method->class);
	    GC_markObject(collector, nop->op_arg.l);
	    if (nop->kind & op_wrapped) {
		int i, len = JREF_TO_ARGVEC(nop->op_arg.l)->len;
		for(i = 0; i < len; ++i)
		    GC_markObject(collector,
				  JREF_TO_ARGVEC(nop->op_arg.l)->args[i]);
	    } else {
		Method *meth = nop->op_id.method;
		int i = 0, len = METHOD_NARGS(meth);
		jvalue *argl = (jvalue*)nop->op_arg.l;
		for(; i < len; ++i, ++argl) {
		    switch (*METHOD_ARG_TYPE(meth, i)) {
		    case 'Z':
		    case 'B':
		    case 'S':
		    case 'C':
		    case 'I':
		    case 'F':
			break;
		    case 'J':
			++argl;
			break;
		    case '[':
		    case 'L':
			GC_markObject(collector, argl->l);
			break;
		    default:
			ABORT();
		    }
		}		
	    }
	    break;

	case op_field_read:
	    GC_markObject(collector, nop->op_id.field_id.clazz);
	    break;

	case op_field_write:
	    GC_markObject(collector, nop->op_id.field_id.clazz);
	    if ((nop->kind & op_wrapped) ||
		!CLASS_PRIM_SIG(FIELD_TYPE(nop->op_id.field_id.field)))
		GC_markObject(collector, nop->op_arg.l);
	    break;

	case op_array_read:
	case op_array_length:
	    break;

	case op_array_write:
	    if ((nop->kind & op_wrapped) ||
		!CLASS_PRIM_SIG(CLASS_ELEMENT_TYPE(OBJECT_CLASS(nop->object))))
		GC_markObject(collector, nop->op_arg.l);
	    break;

	case op_monitor_enter:
	case op_monitor_exit:
	    break;

	default:
	    unimp("walkOperation: unknown operation type");
	    return;
	}
}

