/*
 * xprofiler.h
 * Interface functions to the cross language profiling code
 *
 * Copyright (c) 2000 University of Utah and the Flux Group.
 * All rights reserved.
 *
 * This file is licensed under the terms of the GNU Public License.
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Contributed by the Flux Research Group, Department of Computer Science,
 * University of Utah, http://www.cs.utah.edu/flux/
 */

#ifndef __xprofiler_h
#define __xprofiler_h

#include "mangle.h"

/*
 * Enable profiling
 */
int enableXProfiling(void);
/*
 * Enable call graph profiling, this is separate from enableXProfiling since
 * it can safely be started before the VM.
 */
int enableXCallGraph(void);
/*
 * Disable profiling and deallocate any accounting structures
 */
void disableXProfiling(void);
/*
 * Turn on cross language profiling, this causes hits to be recorded, otherwise
 * they are ignored.  (The default is to record hits).
 */
#if defined(KAFFE_XPROFILER)
void xProfilingOn(void);
#else
#define xProfilingOn()
#endif
/*
 * Turn off cross language profiling, stop recording any hits.
 */
#if defined(KAFFE_XPROFILER)
void xProfilingOff(void);
#else
#define xProfilingOff()
#endif
/*
 * Signal the end of a stage, this will write out gmon files containing the
 * current profiling values and then reset them to zero.
 *
 * stage_name - This should be the name of the just completed stage, it is then
 *   appended to the gmon file name to distinguish from the others.
 */
void xProfileStage(char *stage_name);
/*
 * Set the base name of the generated gmon file(s), defaults to "gmon.out"
 */
int profileGmonFile(char *name);
/*
 * Set the name of the generated assembler file with the profiler symbols.
 * Returns success if the file was created, or a previous call was also
 * successful.
 */
int profileSymbolFile(char *name);
/*
 * Increment the sample count for the given address
 */
void profileHit(char *addr);
/*
 * Increment the count for the given call arc
 */
void profileArcHit(char *frompc, char *selfpc);
/*
 * Start to record hits for the memory covered by this function, and add the
 * functions symbol/address to the symbol file
 */
int profileFunction(struct mangled_method *mm, char *code, int codelen);
/*
 * Start to record hits for the specified segment of memory
 */
int profileMemory(char *code, int codelen);
/*
 * Add the given symbol to the profile symbol file.  (Note: use -1 for unknown
 * size)
 */
int profileSymbol(struct mangled_method *mm, char *addr, int size);
/*
 * Enable the profile timer code
 */
int enableProfileTimer(void);
/*
 * Disable the profile timer code
 */
void disableProfileTimer(void);

/*
 * Indicates whether cross language monitoring code should be added/executed.
 * If this is on, but xProfilingOff() has been called then the appropriate
 * code is still added/executed, but any hits generated by this code isn't
 * recorded.
 */
extern int xProfFlag;

#endif /* __xprofiler_h */
