/**
 * gra.c - 
 *
 * Copyright (c) 1998
 *      Transvirtual Technologies, Inc.  All rights reserved.
 *
 * See the file "license.terms" for information on usage and redistribution 
 * of this file. 
 */

#include "toolkit.h"

typedef struct {
  GC        gc;
  Drawable  drw;
  int       fg;
  int       bg;
  char      xor;
  int       xclr;
  int       x0;
  int       y0;
} Graphics;

#define  WINDOW   0
#define  IMAGE    1
#define  GRAPHICS 2

/*******************************************************************************
 *
 */


void*
Java_java_awt_Toolkit_graInitGraphics ( JNIEnv* env, jclass clazz,
										Graphics* gr, jobject tgt, jint tgtType,
										jint xOff, jint yOff,
										jint xClip, jint yClip, jint wClip, jint hClip,
										jobject fnt, jint fg, jint bg, jboolean blank )
{
  Drawable       drw;
  XGCValues      values;
  XRectangle     rect;
  unsigned long  valueMask = GCForeground | GCBackground | GCFont | GCFunction;

  DBG( awt_gra, ("initGraphics: %x (%x, %d) %d,%d  %d,%d - %d,%d  %x %x %x\n",
				 gr, tgt,tgtType, xOff,yOff, xClip,yClip,wClip,hClip, fnt,fg,bg));

  if ( tgt ) {
	switch ( tgtType ) {
	case 0:  drw = (Drawable)tgt; break;          /* Window   */
	case 1:  drw = ((Image*)tgt)->pix; break;     /* Image    */
	case 2:  drw = ((Graphics*)tgt)->drw; break;  /* Graphics */
	default: fprintf( stderr, "illegal Graphics target: %d\n", tgtType);
		 drw = 0;
	}
  }
  else {
	if ( gr )
	  drw = gr->drw;
	else {
	  fprintf( stderr, "attempt to set an uninitialized graphics");
	  drw = 0;
	}
  }

  values.foreground = fg;
  values.background = bg;
  values.font       = ((XFontStruct*)fnt)->fid;
  values.function   = GXcopy;
  
  if ( gr ) {
	XChangeGC( X->dsp, gr->gc, valueMask, &values);
  }
  else {
	gr = (Graphics*) AWT_MALLOC( sizeof(Graphics));
	gr->gc = XCreateGC( X->dsp, drw, valueMask, &values);

	DBG( awt_gra, (" ->gr: %x (%x)\n", gr, gr->gc));
  }

  rect.x      = xClip; 
  rect.y      = yClip;
  rect.width  = (wClip > 0) ? wClip : 0;
  rect.height = (hClip > 0) ? hClip : 0;
  XSetClipRectangles( X->dsp, gr->gc, xOff, yOff, &rect, 1, Unsorted);

  gr->fg   = fg;
  gr->bg   = bg;
  gr->drw  = drw;
  gr->x0   = xOff;
  gr->y0   = yOff;
  gr->xor  = 0;

  if ( blank ) {
	XSetForeground( X->dsp, gr->gc, gr->bg);
	XFillRectangle( X->dsp, gr->drw, gr->gc, xOff+xClip, yOff+yClip, wClip, hClip);
	XSetForeground( X->dsp, gr->gc, gr->fg);	
  }

  return gr;
}


void
Java_java_awt_Toolkit_graFreeGraphics ( JNIEnv* env, jclass clazz, Graphics* gr )
{
  DBG( awt_gra, ("freeGraphics: %x\n", gr));

  XFreeGC( X->dsp, gr->gc);
  AWT_FREE( gr);
}


void
Java_java_awt_Toolkit_graCopyArea ( JNIEnv* env, jclass clazz, Graphics* gr, 
									jint x, jint y, jint width, jint height,
									jint xDelta, jint yDelta )
{
  DBG( awt_gra, ("copyArea: %x, %d,%d-%d,%d  %d,%d\n", gr, x,y,width,height, xDelta,yDelta));

  x += gr->x0; y += gr->y0;
  XCopyArea( X->dsp, gr->drw, gr->drw, gr->gc, x, y, width, height, x + xDelta, y + yDelta);
  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graClearRect ( JNIEnv* env, jclass clazz, Graphics* gr,
									 jint x, jint y, jint width, jint height )
{
  DBG( awt_gra, ("clearRect: %x, %d,%d-%d,%d\n", gr, x,y,width,height));

  XSetForeground( X->dsp, gr->gc, gr->bg);
  XFillRectangle( X->dsp, gr->drw, gr->gc, x+gr->x0, y+gr->y0, width, height);
  XSetForeground( X->dsp, gr->gc, gr->fg);
  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graDrawBytes ( JNIEnv* env, jclass clazz, Graphics* gr,
									 jarray jBytes, jint offset, jint len, jint x, jint y )
{
  jboolean isCopy;
  int      n;
  jbyte    *jb, *jbo;

  if ( !jBytes ) return;

  n = (*env)->GetArrayLength( env, jBytes);
  jb = (*env)->GetByteArrayElements( env, jBytes, &isCopy);
  jbo = jb + offset;

  DBG( awt_gra, ("drawBytes: %x, %x,%d,%d  \"%s\"  %d,%d\n", gr,
				 jBytes,offset,len, jb, x,y));

  if ( offset+len > n )
	len = n - offset;

  XDrawString( X->dsp, gr->drw, gr->gc, x+gr->x0, y+gr->y0, jbo, len);

  (*env)->ReleaseByteArrayElements( env, jBytes, jb, JNI_ABORT);
  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graDrawChars ( JNIEnv* env, jclass clazz, Graphics* gr,
									 jarray jChars, jint offset, jint len, jint x, jint y )
{
  jboolean isCopy;
  int      n;
  jchar    *jc, *jco;
  XChar2b  *b;

  if ( !jChars ) return;

  n   = (*env)->GetArrayLength( env, jChars);
  jc = (*env)->GetCharArrayElements( env, jChars, &isCopy);
  jco = jc + offset;

  DBG( awt_gra, ("drawChars: %x, %x,%d,%d  \"%s\"  %d,%d\n", gr, jChars,offset,len,
				 jchar2CString(X,jco,len), x,y));

  if ( offset+len > n )
	len = n - offset;

#ifndef WORDS_BIGENDIAN
  n = sizeof(XChar2b)*len;
  b = (XChar2b*) getBuffer( X, n);
  swab( jco, b, n);
#else
  b = (XChar2b*) jco;
#endif

  XDrawString16( X->dsp, gr->drw, gr->gc, x+gr->x0, y+gr->y0, b, len);  

  (*env)->ReleaseCharArrayElements( env, jChars, jc, JNI_ABORT);
  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graDrawString ( JNIEnv* env, jclass clazz,
									  Graphics* gr, jstring str, jint x, jint y )
{
  jboolean     isCopy;
  int          n, len;
  const jchar  *jc;
  XChar2b      *b;

  DBG( awt_gra, ("drawString: %x  \"%s\"  %d,%d\n", gr, java2CString(env,X,str), x,y));

  if ( !str ) return;

  len = (*env)->GetStringLength( env, str);
  jc = (*env)->GetStringChars( env, str, &isCopy);

#ifndef WORDS_BIGENDIAN
  n = sizeof(XChar2b)*len;
  b = (XChar2b*) getBuffer( X, n);
  swab( jc, b, n);
#else
  b = (XChar2b*) jc;
#endif

  XDrawString16( X->dsp, gr->drw, gr->gc, x+gr->x0, y+gr->y0, b, len);  

  (*env)->ReleaseStringChars( env, str, jc);
  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graDrawLine ( JNIEnv* env, jclass clazz, Graphics* gr,
									jint x1, jint y1, jint x2, jint y2 )
{
  DBG( awt_gra, ("drawLine: %x, %d,%d - %d,%d\n", gr, x1,y1, x2,y2));

  if ( (x1==x2) && (y1==y2) ) {  /* swing gimmick */
	XDrawPoint( X->dsp, gr->drw, gr->gc, x1 +gr->x0, y1 +gr->y0);
  }
  else {
	XDrawLine( X->dsp, gr->drw, gr->gc, x1 + gr->x0, y1 + gr->y0, x2 + gr->x0, y2 + gr->y0);
  }
  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graDrawArc ( JNIEnv* env, jclass clazz, Graphics* gr,
								   jint x, jint y, jint width, jint height,
								   jint startAngle, jint arcAngle )
{
  DBG( awt_gra, ("drawArc: %x, %d,%d-%d,%d  %d,%d\n", gr,
				 x,y,width,height, startAngle,arcAngle));

  XDrawArc( X->dsp, gr->drw, gr->gc, x+gr->x0, y+gr->y0, width, height,
			startAngle<<6, arcAngle<<6);
  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graFillArc ( JNIEnv* env, jclass clazz, Graphics* gr,
								   jint x, jint y, jint width, jint height,
								   jint startAngle, jint arcAngle )
{
  DBG( awt_gra, ("fillArc: %x, %d,%d-%d,%d  %d,%d\n", gr,
				 x,y,width,height, startAngle, arcAngle));

  XFillArc( X->dsp, gr->drw, gr->gc, x+gr->x0, y+gr->y0, width, height,
			startAngle<<6, arcAngle<<6);
  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graDrawOval ( JNIEnv* env, jclass clazz, Graphics* gr,
									jint x, jint y, jint width, jint height )
{
  DBG( awt_gra, ("drawOval: %x, %d,%d - %d,%d\n", gr, x,y,width,height));

  XDrawArc( X->dsp, gr->drw, gr->gc, x+gr->x0, y+gr->y0, width, height, 0, 23040);
  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graFillOval ( JNIEnv* env, jclass clazz, Graphics* gr,
									jint x, jint y, jint width, jint height )
{
  DBG( awt_gra, ("fillOval: %x, %d,%d - %d,%d\n", gr, x,y,width,height));

  XFillArc( X->dsp, gr->drw, gr->gc, x+gr->x0, y+gr->y0, width, height, 0, 23040);
  XFLUSH( X, False);
}


int jarray2Points ( JNIEnv* env, Toolkit* X, XPoint** pp, int x0, int y0,
					jarray xPoints, jarray yPoints, int nPoints )
{
  register int i;
  int      n;
  jboolean isCopy;
  jint     *jx = (*env)->GetIntArrayElements( env, xPoints, &isCopy);
  jint     *jy = (*env)->GetIntArrayElements( env, yPoints, &isCopy);

  if ( (n = (*env)->GetArrayLength( env, xPoints)) < nPoints ) nPoints = n;
  if ( (n = (*env)->GetArrayLength( env, yPoints)) < nPoints ) nPoints = n;

  *pp = getBuffer( X, (nPoints+1)*sizeof(XPoint));

  for ( i=0; i<nPoints; i++){
	(*pp)[i].x = jx[i] + x0;
    (*pp)[i].y = jy[i] + y0;
  }

  (*env)->ReleaseIntArrayElements( env, xPoints, jx, JNI_ABORT);
  (*env)->ReleaseIntArrayElements( env, yPoints, jy, JNI_ABORT);

  return nPoints;
}


void
Java_java_awt_Toolkit_graDrawPolygon ( JNIEnv* env, jclass clazz, Graphics* gr,
									   jarray xPoints, jarray yPoints, jint nPoints )
{
  int n;
  XPoint   *p;

  DBG( awt_gra, ("drawPolygon: %x, %x,%x  %d\n", gr, xPoints, yPoints, nPoints));

  if ( !xPoints || !yPoints ) return;

  nPoints = jarray2Points( env, X, &p, gr->x0, gr->y0, xPoints, yPoints, nPoints);

  n = nPoints-1;
  if ( (p[0].x != p[n].x) || (p[0].y != p[n].y) ) {
	p[nPoints].x = p[0].x;
	p[nPoints].y = p[0].y;
	nPoints++;
  }

  XDrawLines( X->dsp, gr->drw, gr->gc, p, nPoints, CoordModeOrigin);
  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graDrawPolyline ( JNIEnv* env, jclass clazz, Graphics* gr,
										jarray xPoints, jarray yPoints, jint nPoints )
{
  XPoint   *p;

  DBG( awt_gra, ("drawPolyline: %x, %x,%x  %d\n", gr, xPoints, yPoints, nPoints));

  if ( !xPoints || !yPoints ) return;

  nPoints = jarray2Points( env, X, &p, gr->x0, gr->y0, xPoints, yPoints, nPoints);
  XDrawLines( X->dsp, gr->drw, gr->gc, p, nPoints, CoordModeOrigin);
  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graFillPolygon ( JNIEnv* env, jclass clazz, Graphics* gr,
									   jarray xPoints, jarray yPoints, jint nPoints )
{
  XPoint   *p;

  DBG( awt_gra, ("fillPolygon: %x, %x,%x  %d\n", gr, xPoints, yPoints, nPoints));

  if ( !xPoints || !yPoints ) return;

  nPoints = jarray2Points( env, X, &p, gr->x0, gr->y0, xPoints, yPoints, nPoints);
  XFillPolygon( X->dsp, gr->drw, gr->gc, p, nPoints, Nonconvex, CoordModeOrigin);
  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graDrawRect ( JNIEnv* env, jclass clazz, Graphics* gr,
									jint x, jint y, jint width, jint height )
{
  DBG( awt_gra, ("drawRect: %x, %d,%d - %d,%d\n", gr, x,y,width,height));

  if ( (width >= 0) && (height >= 0) )
	XDrawRectangle( X->dsp, gr->drw, gr->gc, x+gr->x0, y+gr->y0, width, height);
  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graFillRect ( JNIEnv* env, jclass clazz, Graphics* gr,
									jint x, jint y, jint width, jint height )
{
  DBG( awt_gra, ("fillRect: %x, %d,%d - %d,%d\n", gr, x,y,width,height));

  if ( (width >= 0) && (height >= 0) ) {
	if ( width == 1 ) {   /* some swing gimmicks */
	  XDrawLine( X->dsp, gr->drw, gr->gc, x+gr->x0, y+gr->y0, x+gr->x0, y +gr->y0 +height -1);
	}
	else if ( height == 1 ){
	  XDrawLine( X->dsp, gr->drw, gr->gc, x+gr->x0, y+gr->y0, x +gr->x0 +width -1, y+gr->y0);
	}
	else {
	  XFillRectangle( X->dsp, gr->drw, gr->gc, x+gr->x0, y+gr->y0, width, height);
	}
  }
  XFLUSH( X, False);
}



/*
 * we don't use the Xmu routines here because it would drag the whole Xt libs in
 */
void
Java_java_awt_Toolkit_graDrawRoundRect ( JNIEnv* env, jclass clazz, Graphics* gr,
										 jint x, jint y, jint width, jint height, 
										 jint wArc, jint hArc )
{
  int x1, x2, y1, y2, a, b;

  DBG( awt_gra, ("drawRoundRect: %x, %d,%d - %d,%d  %d,%d\n", gr,
				 x,y,width,height, wArc, hArc));
  
  x += gr->x0;  y += gr->y0;
  a = wArc / 2; b = hArc / 2;

  x1 = x + a;   x2 = x + width - a;  y2 = y + height;
  XDrawLine( X->dsp, gr->drw, gr->gc, x1, y, x2, y);
  XDrawLine( X->dsp, gr->drw, gr->gc, x1, y2, x2, y2);

  y1 = y + b;   y2 = y + height - b; x2 = x + width;
  XDrawLine( X->dsp, gr->drw, gr->gc, x, y1, x, y2);
  XDrawLine( X->dsp, gr->drw, gr->gc, x2, y1, x2, y2);

  XDrawArc( X->dsp, gr->drw, gr->gc, x, y, wArc, hArc, 90*64, 90*64);

  x2 = x + width - wArc;
  XDrawArc( X->dsp, gr->drw, gr->gc, x2, y, wArc, hArc, 0, 90*64);

  y2 = y + height - hArc;
  XDrawArc( X->dsp, gr->drw, gr->gc, x2, y2, wArc, hArc, 0, -90*64);

  XDrawArc( X->dsp, gr->drw, gr->gc, x, y2, wArc, hArc, 180*64, 90*64);

  XFLUSH( X, False);
}

void
Java_java_awt_Toolkit_graFillRoundRect ( JNIEnv* env, jclass clazz, Graphics* gr,
										 jint x, jint y, jint width, jint height, 
										 jint wArc, jint hArc )
{
  int x1, x2, y1, y2, a, b;

  DBG( awt_gra, ("fillRoundRect: %x, %d,%d - %d,%d  %d,%d\n", gr,
				 x,y,width,height, wArc, hArc));
  
  x += gr->x0;  y += gr->y0;
  a = wArc / 2; b = hArc / 2;

  y1 = y + b;  y2 = y + height - b;
  if ( y2 > y1 )
	XFillRectangle( X->dsp, gr->drw, gr->gc, x, y1, width, y2-y1);
  
  x1 = x + a;  x2 = x + width - a;
  if ( x2 > x1 ) {
	XFillRectangle( X->dsp, gr->drw, gr->gc, x1, y, (x2-x1), b);
	XFillRectangle( X->dsp, gr->drw, gr->gc, x1, y2, (x2-x1), b);
  }

  XFillArc( X->dsp, gr->drw, gr->gc, x, y, wArc, hArc, 90*64, 90*64);

  x2 = x + width - wArc;
  XFillArc( X->dsp, gr->drw, gr->gc, x2, y, wArc, hArc, 0, 90*64);

  y2 = y + height - hArc;
  XFillArc( X->dsp, gr->drw, gr->gc, x2, y2, wArc, hArc, 0, -90*64);

  XFillArc( X->dsp, gr->drw, gr->gc, x, y2, wArc, hArc, 180*64, 90*64);

  XFLUSH( X, False);
}


void
Java_java_awt_Toolkit_graDraw3DRect ( JNIEnv* env, jclass clazz, Graphics* gr,
									  jint x, jint y, jint width, jint height,
									  jboolean raised, jint rgb )
{
  int      bright, dark;
  int      xw, yh;

  DBG( awt_gra, ("draw3DRect: %x, %d,%d - %d,%d  %d %x\n", gr,
				 x,y,width,height, raised, rgb));

  /* we should use the same mechanism like ordinary Graphics brighter()/darker() here */
  dark   = (int) (Java_java_awt_Toolkit_clrDark( env, clazz, rgb) >> 32);
  bright = (int) (Java_java_awt_Toolkit_clrBright( env, clazz, rgb) >> 32);

  x += gr->x0; y += gr->y0;
  xw = x + width;
  yh = y + height;

  XSetForeground( X->dsp, gr->gc, (raised ? bright : dark));
  XDrawLine( X->dsp, gr->drw, gr->gc, x, y, xw-1, y);
  XDrawLine( X->dsp, gr->drw, gr->gc, x, y, x, yh);

  XSetForeground( X->dsp, gr->gc, (raised ? dark : bright));
  XDrawLine( X->dsp, gr->drw, gr->gc, x+1, yh, xw, yh);
  XDrawLine( X->dsp, gr->drw, gr->gc, xw, y, xw, yh);

  XSetForeground( X->dsp, gr->gc, gr->fg);

  XFLUSH( X, False);
}

void
Java_java_awt_Toolkit_graFill3DRect ( JNIEnv* env, jclass clazz, Graphics* gr,
									  jint x, jint y, jint width, jint height,
									  jboolean raised, jint rgb )
{
  DBG( awt_gra, ("fill3DRect: %x, %d,%d - %d,%d  %d %x\n", gr,
				 x,y,width,height, raised, rgb));

  XFillRectangle( X->dsp, gr->drw, gr->gc, x+gr->x0+1, y+gr->y0+1, width-2, height-2);

  Java_java_awt_Toolkit_graDraw3DRect( env, clazz, gr, x, y, width-1, height-1, raised, rgb);
}


void
Java_java_awt_Toolkit_graAddClip ( JNIEnv* env, jclass clazz, Graphics* gr,
																	jint xClip, jint yClip, jint wClip, jint hClip )
{
	/* not implemented yet */
}


void
Java_java_awt_Toolkit_graSetClip ( JNIEnv* env, jclass clazz, Graphics* gr,
								   jint xClip, jint yClip, jint wClip, jint hClip )
{
  XRectangle rect;

  DBG( awt_gra, ("setClip: %x, %d,%d - %d,%d\n", gr, xClip, yClip, wClip, hClip));

  rect.x      = xClip;
  rect.y      = yClip;
  rect.width  = (wClip > 0) ? wClip : 0;
  rect.height = (hClip > 0) ? hClip : 0;
  XSetClipRectangles( X->dsp, gr->gc, gr->x0, gr->y0, &rect, 1, Unsorted);
}

void
Java_java_awt_Toolkit_graSetColor ( JNIEnv* env, jclass clazz, Graphics* gr, jint clr )
{
  DBG( awt_gra, ("setColor: %x, %x\n", gr, clr));

  gr->fg = clr;

  if ( gr->xor )
	clr ^= gr->xclr;

  XSetForeground( X->dsp, gr->gc, clr);
}

void
Java_java_awt_Toolkit_graSetBackColor ( JNIEnv* env, jclass clazz, Graphics* gr, jint clr )
{
  DBG( awt_gra, ("setBackColor: %x, %x\n", gr, clr));

  gr->bg = clr;
  XSetBackground( X->dsp, gr->gc, clr);
}

void
Java_java_awt_Toolkit_graSetFont ( JNIEnv* env, jclass clazz, Graphics* gr, jobject fnt )
{
  DBG( awt_gra, ("setFont: %x, %x\n", gr, fnt));

  XSetFont( X->dsp, gr->gc, ((XFontStruct*)fnt)->fid);
}

void
Java_java_awt_Toolkit_graSetOffset ( JNIEnv* env, jclass clazz, Graphics* gr, jint xOff, jint yOff )
{
  DBG( awt_gra, ("setOffset: %x, %d,%d\n", gr, xOff, yOff));

  gr->x0 = xOff;
  gr->y0 = yOff;
}

void
Java_java_awt_Toolkit_graSetPaintMode ( JNIEnv* env, jclass clazz, Graphics* gr )
{
  DBG( awt_gra, ("setPaintMode: %x\n", gr));

  gr->xor = 0;
  XSetForeground( X->dsp, gr->gc, gr->fg);
  XSetFunction( X->dsp, gr->gc, GXcopy);
}

void
Java_java_awt_Toolkit_graSetXORMode ( JNIEnv* env, jclass clazz, Graphics* gr, jint xorClr )
{
  DBG( awt_gra, ("setXORMode: %x, %x\n", gr, xorClr));

  gr->xor = 1;
  gr->xclr = xorClr;
  XSetForeground( X->dsp, gr->gc, gr->fg ^ xorClr);
  XSetFunction( X->dsp, gr->gc, GXxor);
}

void
Java_java_awt_Toolkit_graSetVisible ( JNIEnv* env, jclass clazz, Graphics* gr, jint isVisible )
{
  DBG( awt_gra, ("setVisble: %x  %d\n", gr, isVisible));

  /*
   * This is rather a hack to "defuse" a Graphics object, but we don't want to
   * add checks on every draw op to test if the target is invisible
   */
  if ( !isVisible ) {
	if ( gr->y0 >= -10000 ) 
	  gr->y0 -= 100000;
  }
  else {
	if ( gr->y0 <= -10000 )
	  gr->y0 += 100000;
  }
}


/************************************************************************************
 * image rendering
 */

void
drawAlphaImage ( Graphics* gr, Image* img,
				 jint srcX, jint srcY, jint dstX, jint dstY,
				 jint width, jint height, jint bgval )
{
  Image  *dstImg = 0;
  XImage *dstXim = 0;
  int    i, j, si, sj, alpha;
  unsigned long dpix, spix, bgpix = 0;
  int    sr, sg, sb, dr, dg, db;

  DBG( awt_gra, ("drawAlphaImage: %x %x (%x, %x),  %d,%d  %d,%d  %d,%d  %x\n",
				 gr, img, img->xImg, img->alpha, srcX, srcY, dstX, dstY, width, height, bgval));

#ifdef HAVE_LIBXEXT
  if ( X->shm == USE_SHM ){
	dstImg = createImage( width, height);
	createXImage( X, dstImg);
	if ( dstImg->shmiImg ){
	  dstXim = dstImg->xImg;
	  XShmGetImage( X->dsp, gr->drw, dstXim, dstX, dstY, 0xffffffff);
	}
	else {  /* Shm failed, backup to normal XImage */
#endif
	  Java_java_awt_Toolkit_imgFreeImage( 0, 0, dstImg);
	  dstImg = 0;
#ifdef HAVE_LIBXEXT
	}
  }
#endif

  if ( !dstXim ) {
	dstXim = XGetImage( X->dsp, gr->drw, dstX, dstY, width, height, 0xffffffff, ZPixmap);
  }

  if ( bgval != -1 )
	bgpix = pixelValue( X, bgval);

  if ( dstXim ) {
	for ( j=0, sj=srcY; j<dstXim->height; j++, sj++ ) {
	  for ( i=0, si=srcX; i<dstXim->width; i++, si++ ) {
		dpix = (bgval == -1) ? XGetPixel( dstXim, i, j) : bgpix;
		spix = XGetPixel( img->xImg, si, sj);
		alpha = GetAlpha( img->alpha, si, sj);

		rgbValues( X, dpix, &dr, &dg, &db);

		if ( alpha ) {
		  /*
		   * ints still considered to be substantially faster than floats on most
		   * architectures (avoid div by 255), map all 0..255 -> 1..256
		   */
		  alpha++; sr++; sg++; sb++; dr++; dg++; db++;

		  rgbValues( X, spix, &sr, &sg, &sb);
		  dr = ((alpha * sr + (256 - alpha) * dr) + 128) >> 8;
		  dg = ((alpha * sg + (256 - alpha) * dg) + 128) >> 8;
		  db = ((alpha * sb + (256 - alpha) * db) + 128) >> 8;

		  dr--; dg--; db--;

		  XPutPixel( dstXim, i, j, pixelValue( X, (dr << 16)|(dg << 8)|(db) ));
		}
	  }
	}

#ifdef HAVE_LIBXEXT
	if ( dstImg != 0 ){
	  XShmPutImage( X->dsp, gr->drw, gr->gc, dstXim, 0, 0, dstX, dstY, width, height, False);
	  XSync( X->dsp, False);
	  Java_java_awt_Toolkit_imgFreeImage( 0, 0, dstImg);
	}
	else {
#endif
	  XPutImage( X->dsp, gr->drw, gr->gc, dstXim, 0, 0, dstX, dstY, width, height);
	  XFlush( X->dsp);
	  XDestroyImage( dstXim);
#ifdef HAVE_LIBXEXT
	}
#endif

	XFLUSH( X, True);
  }
}


void
Java_java_awt_Toolkit_graDrawImage ( JNIEnv* env, jclass clazz, Graphics* gr, Image* img,
									 jint srcX, jint srcY,
									 jint dstX, jint dstY,
									 jint width, jint height, jint bgval )
{
  XGCValues values;

  DBG( awt_gra, ("drawImage: %x %x (%x,%x,%x %d,%d) %d,%d, %d,%d, %d,%d, %x\n",
				 gr, img,  img->xImg,img->xMask,img->alpha,  img->width,img->height,
				 srcX,srcY,  dstX,dstY,	 width,height, bgval));

  dstX += gr->x0;
  dstY += gr->y0;

  if ( img->pix ) {
	XCopyArea( X->dsp, img->pix, gr->drw, gr->gc, srcX, srcY, width, height, dstX, dstY);
  }
  else if ( img->xImg ) {
	if ( img->xMask ) {
	  /* save current gc values, we have to change them temporarily */
	  XGetGCValues( X->dsp, gr->gc, GCFunction|GCPlaneMask|GCForeground|GCBackground,
					&values);

	  /* in case we have a specified background color, fill area */
	  if ( bgval != -1 ) {
		XSetForeground( X->dsp, gr->gc, bgval);
		XFillRectangle( X->dsp, gr->drw, gr->gc, dstX, dstY, width, height);
	  }

	  /* set attributes so that a subsequent image draw leaves the mask pels unmodified */
	  XSetFunction( X->dsp, gr->gc, GXandInverted);
	  XSetForeground( X->dsp, gr->gc, 0xffffffff);
	  XSetBackground( X->dsp, gr->gc, 0);

	  /* draw the mask bitmap */
#ifdef HAVE_LIBXEXT
	  if ( img->shmiMask ){
		XShmPutImage( X->dsp, gr->drw, gr->gc, img->xMask,
					  srcX, srcY, dstX, dstY, width, height, False);
	  }
	  else {
#endif
		XPutImage( X->dsp, gr->drw, gr->gc, img->xMask,
				   srcX, srcY, dstX, dstY, width, height);
#ifdef HAVE_LIBXEXT
	  }
#endif

	  /* restore gc values except of the function */
	  XChangeGC( X->dsp, gr->gc, GCFunction|GCPlaneMask|GCForeground|GCBackground, &values);
	  XSetFunction( X->dsp, gr->gc, GXor);
	}
	else if ( img->alpha ) {
	  drawAlphaImage( gr, img, srcX, srcY, dstX, dstY, width, height, bgval);
	  return;
	}

	/* this is the real image drawing */
#ifdef HAVE_LIBXEXT
	if ( img->shmiImg ){
	  XShmPutImage( X->dsp, gr->drw, gr->gc, img->xImg,
					srcX, srcY, dstX, dstY, width, height, False);
	}
	else {
#endif
	  XPutImage( X->dsp, gr->drw, gr->gc, img->xImg,
				 srcX, srcY, dstX, dstY, width, height);
#ifdef HAVE_LIBXEXT
	}
#endif

	/* in case we had a mask, restore the original function */
	if ( img->xMask )
	  XSetFunction( X->dsp, gr->gc, values.function);
  }

  XFLUSH( X, True);
}


void
Java_java_awt_Toolkit_graDrawImageScaled ( JNIEnv* env, jclass clazz, Graphics* gr, Image* img,
										   jint dx0, jint dy0, jint dx1, jint dy1,
										   jint sx0, jint sy0, jint sx1, jint sy1, jint bgval )
{
  int        iw, x0, y0, x1, y1;
  Image      *tgt;
  int        tmpXImg = (img->xImg == NULL);

  DBG( awt_gra, ("drawImageScaled: %x %x (%x), %d,%d,%d,%d, %d,%d,%d,%d, %x\n",
				 gr, img, img->xImg, dx0, dy0, dx1, dy1, sx0, sy0, sx1, sy1, bgval));

  /* screen images don't have an XImage, we have to get the data first */
  if ( tmpXImg ) {
  	img->xImg = XGetImage( X->dsp, img->pix, 0, 0, img->width, img->height,
						   0xffffffff, ZPixmap);
  }

  iw = img->xImg->width;

  if ( dx1 > dx0 ) {
	x0 = dx0; x1 = dx1;
  }
  else {
	x0 = dx1; x1 = dx0;
  }
  if ( dy1 > dy0 ) {
	y0 = dy0; y1 = dy1;
  }
  else {
	y0 = dy1; y1 = dy0;
  }

  if ( sx0 < 0 ) sx0 = 0;
  if ( sx1 < 0 ) sx1 = 0;
  if ( sx0 >= iw ) sx0 = iw-1;
  if ( sx1 >= iw ) sx1 = iw-1;

  tgt = createImage( (x1 - x0 +1), (y1 - y0 +1));
  /* alpha image should always go first, since it would disable subsequent Shm usage */
  if ( img->alpha )
	createAlphaImage( X, tgt);
  if ( img->xMask )
	createXMaskImage( X, tgt);
  createXImage( X, tgt);

  initScaledImage( X, tgt, img, dx0-x0, dy0-y0, dx1-x0, dy1-y0, sx0, sy0, sx1, sy1);
  Java_java_awt_Toolkit_graDrawImage ( env, clazz, gr, tgt, 0, 0,
									   x0, y0, tgt->width, tgt->height, bgval);
#ifdef HAVE_LIBXEXT
  if ( tgt->shmiImg ) {
	XSync( X->dsp, False); /* since we're going to destroy tgt, process its drawing first */
  }
#endif
  Java_java_awt_Toolkit_imgFreeImage( env, clazz, tgt);

  if ( tmpXImg ) {
	XDestroyImage( img->xImg);
	img->xImg = NULL;
  }
}
